const { ipcRenderer } = require('electron');

// Since contextIsolation is false, we can directly attach to window
// Make sure to initialize immediately
window.electronAPI = {
  showSaveDialog: () => ipcRenderer.invoke('show-save-dialog'),
  showOpenDialog: () => ipcRenderer.invoke('show-open-dialog'),
  writeFile: (fileName, data) => ipcRenderer.invoke('write-file', fileName, data),
  readFile: (fileName) => ipcRenderer.invoke('read-file', fileName),
  showSaveDialogWithDefault: (defaultPath) => ipcRenderer.invoke('show-save-dialog-with-default', defaultPath)
};

// Debug log to confirm API is loaded
console.log('electronAPI loaded:', window.electronAPI);

// Safely require modules with fallbacks
function safeRequire(moduleName, fallback = null) {
  try {
    return require(moduleName);
  } catch (e) {
    console.warn(`Module ${moduleName} not found, using fallback`);
    return fallback;
  }
}

// Expose Node.js modules that the renderer needs
window.nodeAPI = {
  atob: safeRequire('atob', (str) => Buffer.from(str, 'base64').toString('binary')),
  btoa: safeRequire('btoa', (str) => Buffer.from(str, 'binary').toString('base64')),
  crc16modbus: (() => {
    const crc = safeRequire('crc');
    return crc ? crc.crc16modbus : null;
  })(),
  os: safeRequire('os'),
  serialport: safeRequire('serialport'),
  httpclient: safeRequire('httpclient'),
  fs: safeRequire('fs'),
  path: safeRequire('path'),
  request: safeRequire('request')
};

// Expose jQuery directly to the global scope
try {
  const jQuery = require('../node_modules/jquery/dist/jquery.min.js');
  // Make jQuery available globally immediately
  window.$ = jQuery;
  window.jQuery = jQuery;
} catch (e) {
  console.error('Failed to load jQuery:', e);
  // Fallback: expose a placeholder that will be replaced when jQuery loads
  window.$ = undefined;
  window.jQuery = undefined;
}

// Load Bootstrap and Bootstrap Toggle after DOM is ready
window.addEventListener('DOMContentLoaded', () => {
  // Wait a bit for jQuery to be available, then load Bootstrap
  setTimeout(() => {
    const bootstrapScript = document.createElement('script');
    bootstrapScript.src = './node_modules/bootstrap/dist/js/bootstrap.min.js';
    document.head.appendChild(bootstrapScript);
    
    bootstrapScript.onload = () => {
      // Load Bootstrap Toggle after Bootstrap
      const toggleScript = document.createElement('script');
      toggleScript.src = './node_modules/bootstrap-toggle/js/bootstrap-toggle.js';
      document.head.appendChild(toggleScript);
    };
  }, 100);
});